<?php
/**
 * Content processing engine
 * Applies automatic links to post content
 *
 * @package MaxBes_Auto_Link_Manager
 */

if (!defined('ABSPATH')) {
    exit;
}

class MaxBes_AutoLink_Processor {

    /**
     * Process post content and apply links
     *
     * @param string $content Post content
     * @return string Modified content
 */
    public function process_content($content) {
        global $post;

        // Skip if not in main loop or no post
        if (!is_main_query() || !in_the_loop() || !$post) {
            return $content;
        }

        // Skip if post type not selected
        if (!$this->should_process_post_type($post->post_type)) {
            return $content;
        }

        // Get keywords
        $keywords = $this->get_keywords_for_processing();

        if (empty($keywords)) {
            return $content;
        }

        // Apply links
        $content = $this->apply_keywords($content, $keywords, $post->ID);

        return $content;
    }

    /**
     * Check if post type should be processed
     *
     * @param string $post_type Post type
     * @return bool
     */
    private function should_process_post_type($post_type) {
        $settings = maxbes_autolink_get_settings();
        $selected_types = isset($settings['selected_post_types']) ? $settings['selected_post_types'] : array('post', 'page');

        return in_array($post_type, $selected_types);
    }

    /**
     * Get keywords for processing
     *
     * @return array
     */
    private function get_keywords_for_processing() {
        // Load from database
        $keywords = MaxBes_AutoLink_Database::get_active_keywords();

        // Lite version: Sort by length only (longer keywords first)
        usort($keywords, function($a, $b) {
            return strlen($b->keyword) - strlen($a->keyword);
        });

        return $keywords;
    }

    /**
     * Apply keywords to content
     *
     * @param string $content Content
     * @param array $keywords Keywords
     * @param int $post_id Post ID
     * @return string
     */
    private function apply_keywords($content, $keywords, $post_id) {
        $settings = maxbes_autolink_get_settings();
        // Lite version: Hard-coded limit of 5 links per post
        $max_links_per_post = MAXBES_AUTOLINK_MAX_LINKS_PER_POST;

        $total_links_applied = 0;
        $linked_keywords = array();

        foreach ($keywords as $keyword_obj) {
            // Stop if max links reached
            if ($total_links_applied >= $max_links_per_post) {
                break;
            }

            $keyword = $keyword_obj->keyword;
            $url = $keyword_obj->url;
            $max_links = (int) $keyword_obj->max_links;
            $case_sensitive = (bool) $keyword_obj->case_sensitive;

            // Skip if already linked
            $check_key = strtolower($keyword);
            if (in_array($check_key, $linked_keywords)) {
                continue;
            }

            // Build regex pattern
            $pattern = $this->build_regex_pattern($keyword, $case_sensitive);

            // Count how many times we can link this keyword
            $links_to_apply = min($max_links, $max_links_per_post - $total_links_applied);

            // Apply the link
            $replacement = $this->build_replacement_html($url, $settings);
            $new_content = preg_replace($pattern, $replacement, $content, $links_to_apply, $count);

            if ($count > 0) {
                $content = $new_content;
                $total_links_applied += $count;
                $linked_keywords[] = $check_key;
            }
        }

        return $content;
    }

    /**
     * Build regex pattern for keyword matching
     *
     * @param string $keyword Keyword
     * @param bool $case_sensitive Case sensitive
     * @return string Regex pattern
     */
    private function build_regex_pattern($keyword, $case_sensitive) {
        // Lite version: Fixed protected tags
        $protected_tags = array('h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'a', 'code', 'pre', 'figcaption');

        $escaped_keyword = preg_quote($keyword, '/');

        // Build negative lookahead for protected tags
        $protected_pattern = '';
        if (!empty($protected_tags)) {
            $tags_pattern = implode('|', array_map('preg_quote', $protected_tags));
            $protected_pattern = '(?![^<]*>|[^<>]*<\/(' . $tags_pattern . ')>)';
        }

        // Word boundary + keyword + word boundary + not inside tags/links
        $pattern = '/\b(' . $escaped_keyword . ')\b' . $protected_pattern . '/';

        // Add case insensitive flag if needed
        if (!$case_sensitive) {
            $pattern .= 'iu'; // i = case insensitive, u = UTF-8
        } else {
            $pattern .= 'u';
        }

        return $pattern;
    }

    /**
     * Build replacement HTML
     *
     * @param string $url URL
     * @param array $settings Settings (not used in lite version)
     * @return string
     */
    private function build_replacement_html($url, $settings) {
        // Lite version: Fixed link attributes
        $link_class = 'maxbes-auto-link';
        $link_target = '_self';

        $attributes = array(
            'href="' . esc_url($url) . '"',
            'class="' . $link_class . '"',
        );

        return '<a ' . implode(' ', $attributes) . '>$1</a>';
    }
}
