<?php
/**
 * Database operations class
 * Handles all CRUD operations for keywords table
 *
 * @package MaxBes_Auto_Link_Manager
 */

if (!defined('ABSPATH')) {
    exit;
}

class MaxBes_AutoLink_Database {

    /**
     * Get table name
     */
    private static function get_table_name() {
        global $wpdb;
        return $wpdb->prefix . MAXBES_AUTOLINK_TABLE;
    }

    /**
     * Insert new keyword
     *
     * @param array $data Keyword data
     * @return int|false Inserted ID or false on failure
     */
    public static function insert_keyword($data) {
        global $wpdb;

        // Lite version: Check keyword limit (50 max)
        $current_count = self::get_keywords_count('all');
        if ($current_count >= MAXBES_AUTOLINK_MAX_KEYWORDS) {
            return false;
        }

        $defaults = array(
            'keyword' => '',
            'url' => '',
            'link_type' => 'taxonomy',
            'source_type' => null,
            'source_id' => null,
            'priority' => 3, // Lite version: fixed priority
            'max_links' => 1,
            'case_sensitive' => 0,
            'status' => 'active',
        );

        $data = wp_parse_args($data, $defaults);

        // Sanitize
        $data['keyword'] = sanitize_text_field($data['keyword']);
        $data['url'] = esc_url_raw($data['url']);
        $data['link_type'] = sanitize_text_field($data['link_type']);
        $data['source_type'] = sanitize_text_field($data['source_type']);
        $data['priority'] = absint($data['priority']);
        $data['max_links'] = absint($data['max_links']);
        $data['case_sensitive'] = absint($data['case_sensitive']);
        $data['status'] = sanitize_text_field($data['status']);

        $result = $wpdb->insert(
            self::get_table_name(),
            $data,
            array('%s', '%s', '%s', '%s', '%d', '%d', '%d', '%d', '%s')
        );

        return $result ? $wpdb->insert_id : false;
    }

    /**
     * Update keyword
     *
     * @param int $id Keyword ID
     * @param array $data Keyword data
     * @return bool
     */
    public static function update_keyword($id, $data) {
        global $wpdb;

        // Sanitize
        if (isset($data['keyword'])) $data['keyword'] = sanitize_text_field($data['keyword']);
        if (isset($data['url'])) $data['url'] = esc_url_raw($data['url']);
        if (isset($data['link_type'])) $data['link_type'] = sanitize_text_field($data['link_type']);
        if (isset($data['source_type'])) $data['source_type'] = sanitize_text_field($data['source_type']);
        if (isset($data['priority'])) $data['priority'] = absint($data['priority']);
        if (isset($data['max_links'])) $data['max_links'] = absint($data['max_links']);
        if (isset($data['case_sensitive'])) $data['case_sensitive'] = absint($data['case_sensitive']);
        if (isset($data['status'])) $data['status'] = sanitize_text_field($data['status']);

        $result = $wpdb->update(
            self::get_table_name(),
            $data,
            array('id' => $id),
            null,
            array('%d')
        );

        return $result !== false;
    }

    /**
     * Delete keyword
     *
     * @param int $id Keyword ID
     * @return bool
     */
    public static function delete_keyword($id) {
        global $wpdb;

        $result = $wpdb->delete(
            self::get_table_name(),
            array('id' => $id),
            array('%d')
        );

        return $result !== false;
    }

    /**
     * Get single keyword by ID
     *
     * @param int $id Keyword ID
     * @return object|null
     */
    public static function get_keyword($id) {
        global $wpdb;

        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM " . self::get_table_name() . " WHERE id = %d",
            $id
        ));
    }

    /**
     * Get all keywords
     *
     * @param array $args Query arguments
     * @return array
     */
    public static function get_all_keywords($args = array()) {
        global $wpdb;

        $defaults = array(
            'status' => 'all',
            'link_type' => 'all',
            'source_type' => 'all',
            'orderby' => 'priority',
            'order' => 'ASC',
            'limit' => 0,
            'offset' => 0,
        );

        $args = wp_parse_args($args, $defaults);

        $where = array();

        // Status filter
        if ($args['status'] !== 'all') {
            $where[] = $wpdb->prepare("status = %s", $args['status']);
        }

        // Link type filter
        if ($args['link_type'] !== 'all') {
            $where[] = $wpdb->prepare("link_type = %s", $args['link_type']);
        }

        // Source type filter
        if ($args['source_type'] !== 'all') {
            $where[] = $wpdb->prepare("source_type = %s", $args['source_type']);
        }

        $where_sql = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';

        // Build query
        $sql = "SELECT * FROM " . self::get_table_name() . " $where_sql";

        // Order by
        $sql .= " ORDER BY {$args['orderby']} {$args['order']}";

        // Limit
        if ($args['limit'] > 0) {
            $sql .= $wpdb->prepare(" LIMIT %d OFFSET %d", $args['limit'], $args['offset']);
        }

        return $wpdb->get_results($sql);
    }

    /**
     * Get active keywords only (for processing)
     *
     * @return array
     */
    public static function get_active_keywords() {
        return self::get_all_keywords(array(
            'status' => 'active',
            'orderby' => 'priority',
            'order' => 'ASC'
        ));
    }

    /**
     * Search keywords
     *
     * @param string $search_term Search term
     * @return array
     */
    public static function search_keywords($search_term) {
        global $wpdb;

        $search_term = '%' . $wpdb->esc_like($search_term) . '%';

        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM " . self::get_table_name() . "
            WHERE keyword LIKE %s OR url LIKE %s
            ORDER BY priority ASC",
            $search_term,
            $search_term
        ));
    }

    /**
     * Check if keyword exists
     *
     * @param string $keyword Keyword text
     * @param int $exclude_id ID to exclude (for updates)
     * @return bool
     */
    public static function keyword_exists($keyword, $exclude_id = 0) {
        global $wpdb;

        $sql = $wpdb->prepare(
            "SELECT COUNT(*) FROM " . self::get_table_name() . " WHERE keyword = %s",
            $keyword
        );

        if ($exclude_id > 0) {
            $sql .= $wpdb->prepare(" AND id != %d", $exclude_id);
        }

        return $wpdb->get_var($sql) > 0;
    }

    /**
     * Get keywords count
     *
     * @param string $status Status filter
     * @return int
     */
    public static function get_keywords_count($status = 'all') {
        global $wpdb;

        $where = $status !== 'all' ? $wpdb->prepare("WHERE status = %s", $status) : '';

        return (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM " . self::get_table_name() . " $where"
        );
    }

    /**
     * Get keywords paginated
     *
     * @param int $page Page number
     * @param int $per_page Items per page
     * @param array $filters Filters
     * @return array
     */
    public static function get_keywords_paginated($page = 1, $per_page = 20, $filters = array()) {
        $offset = ($page - 1) * $per_page;

        $args = array_merge($filters, array(
            'limit' => $per_page,
            'offset' => $offset
        ));

        return self::get_all_keywords($args);
    }

    /**
     * Bulk delete keywords
     *
     * @param array $ids Array of keyword IDs
     * @return int Number of deleted rows
     */
    public static function bulk_delete($ids) {
        global $wpdb;

        if (empty($ids)) {
            return 0;
        }

        $ids = array_map('absint', $ids);
        $placeholders = implode(',', array_fill(0, count($ids), '%d'));

        return $wpdb->query($wpdb->prepare(
            "DELETE FROM " . self::get_table_name() . " WHERE id IN ($placeholders)",
            $ids
        ));
    }

    /**
     * Bulk update status
     *
     * @param array $ids Array of keyword IDs
     * @param string $status New status
     * @return int Number of updated rows
     */
    public static function bulk_update_status($ids, $status) {
        global $wpdb;

        if (empty($ids)) {
            return 0;
        }

        $ids = array_map('absint', $ids);
        $placeholders = implode(',', array_fill(0, count($ids), '%d'));

        return $wpdb->query($wpdb->prepare(
            "UPDATE " . self::get_table_name() . " SET status = %s WHERE id IN ($placeholders)",
            array_merge(array($status), $ids)
        ));
    }

    /**
     * Get statistics
     *
     * @return array
     */
    public static function get_statistics() {
        global $wpdb;
        $table = self::get_table_name();

        return array(
            'total' => self::get_keywords_count('all'),
            'active' => self::get_keywords_count('active'),
            'inactive' => self::get_keywords_count('inactive'),
            'by_type' => $wpdb->get_results(
                "SELECT link_type, COUNT(*) as count FROM $table GROUP BY link_type"
            ),
            'by_source' => $wpdb->get_results(
                "SELECT source_type, COUNT(*) as count FROM $table WHERE source_type IS NOT NULL GROUP BY source_type"
            )
        );
    }
}
