<?php
/**
 * Cache management class
 * Handles all caching operations for the plugin
 *
 * @package MaxBes_Auto_Link_Manager
 */

if (!defined('ABSPATH')) {
    exit;
}

class MaxBes_AutoLink_Cache {

    /**
     * Get cache duration from settings
     *
     * @return int Cache duration in seconds
     */
    private static function get_cache_duration() {
        $settings = get_option('maxbes_autolink_settings', array());
        return isset($settings['cache_duration']) ? (int) $settings['cache_duration'] : DAY_IN_SECONDS;
    }

    /**
     * Get cached keywords list
     *
     * @return array|false Keywords array or false if not cached
     */
    public static function get_keywords_cache() {
        return get_transient('maxbes_autolink_keywords_list');
    }

    /**
     * Set keywords list cache
     *
     * @param array $keywords Keywords array
     * @return bool
     */
    public static function set_keywords_cache($keywords) {
        return set_transient(
            'maxbes_autolink_keywords_list',
            $keywords,
            self::get_cache_duration()
        );
    }

    /**
     * Clear keywords list cache
     *
     * @return bool
     */
    public static function clear_keywords_cache() {
        return delete_transient('maxbes_autolink_keywords_list');
    }

    /**
     * Get processed post content cache
     *
     * @param int $post_id Post ID
     * @return string|false Processed content or false if not cached
     */
    public static function get_post_cache($post_id) {
        return get_transient('maxbes_autolink_post_' . $post_id);
    }

    /**
     * Set processed post content cache
     *
     * @param int $post_id Post ID
     * @param string $content Processed content
     * @return bool
     */
    public static function set_post_cache($post_id, $content) {
        return set_transient(
            'maxbes_autolink_post_' . $post_id,
            $content,
            self::get_cache_duration()
        );
    }

    /**
     * Clear single post cache
     *
     * @param int $post_id Post ID
     * @return bool
     */
    public static function clear_post_cache($post_id) {
        return delete_transient('maxbes_autolink_post_' . $post_id);
    }

    /**
     * Clear post cache when post is saved
     *
     * @param int $post_id Post ID
     */
    public static function clear_post_cache_on_save($post_id) {
        // Check if this is an autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        // Check user permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        // Clear this post's cache
        self::clear_post_cache($post_id);
    }

    /**
     * Clear all plugin caches
     *
     * @return bool
     */
    public static function clear_all_cache() {
        global $wpdb;

        // Delete all transients starting with maxbes_autolink_
        $result = $wpdb->query(
            "DELETE FROM {$wpdb->options}
            WHERE option_name LIKE '_transient_maxbes_autolink_%'
            OR option_name LIKE '_transient_timeout_maxbes_autolink_%'"
        );

        return $result !== false;
    }

    /**
     * Get cache statistics
     *
     * @return array Cache stats
     */
    public static function get_cache_stats() {
        global $wpdb;

        // Count cached items
        $total_cached = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->options}
            WHERE option_name LIKE '_transient_maxbes_autolink_%'
            AND option_name NOT LIKE '_transient_timeout_%'"
        );

        // Get cache size (approximate)
        $cache_size = $wpdb->get_var(
            "SELECT SUM(LENGTH(option_value)) FROM {$wpdb->options}
            WHERE option_name LIKE '_transient_maxbes_autolink_%'
            AND option_name NOT LIKE '_transient_timeout_%'"
        );

        return array(
            'total_items' => (int) $total_cached,
            'total_size' => size_format((int) $cache_size),
            'keywords_cached' => (bool) self::get_keywords_cache(),
            'cache_duration' => human_time_diff(0, self::get_cache_duration())
        );
    }

    /**
     * Warm up cache - Preload commonly used data
     */
    public static function warmup_cache() {
        // Preload keywords list
        if (!self::get_keywords_cache()) {
            $keywords = MaxBes_AutoLink_Database::get_active_keywords();
            self::set_keywords_cache($keywords);
        }
    }
}
