<?php
/**
 * Fired during plugin activation
 *
 * @package MaxBes_Auto_Link_Manager
 */

if (!defined('ABSPATH')) {
    exit;
}

class MaxBes_AutoLink_Activator {

    /**
     * Activate the plugin
     * - Create database table
     * - Set default settings
     */
    public static function activate() {
        global $wpdb;

        $table_name = $wpdb->prefix . MAXBES_AUTOLINK_TABLE;
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            keyword varchar(255) NOT NULL,
            url varchar(500) NOT NULL,
            link_type varchar(50) DEFAULT 'taxonomy' COMMENT 'taxonomy, post, page, custom',
            source_type varchar(100) DEFAULT NULL COMMENT 'category, post_tag, property-city, post, page, etc',
            source_id bigint(20) DEFAULT NULL COMMENT 'Term ID or Post ID',
            priority tinyint(2) DEFAULT 5 COMMENT '1=highest, 10=lowest',
            max_links tinyint(2) DEFAULT 1 COMMENT 'Max occurrences per post',
            case_sensitive tinyint(1) DEFAULT 0 COMMENT '0=no, 1=yes',
            status varchar(20) DEFAULT 'active' COMMENT 'active, inactive',
            click_count bigint(20) DEFAULT 0 COMMENT 'Optional: track clicks',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

            UNIQUE KEY unique_keyword (keyword),
            KEY idx_status (status),
            KEY idx_priority (priority),
            KEY idx_link_type (link_type),
            KEY idx_source (source_type, source_id)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);

        // Set default settings
        $default_settings = array(
            'enabled' => true,
            'max_links_per_post' => 12,
            'case_sensitive' => false,
            'protected_tags' => array('h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'a', 'code', 'pre', 'figcaption'),
            'cache_duration' => DAY_IN_SECONDS,
            'link_target' => '_self',
            'link_class' => 'maxbes-auto-link',
            'rel_attribute' => '',
            'selected_taxonomies' => array(), // User will select which taxonomies to use
            'selected_post_types' => array('post', 'page'), // Default post types
        );

        // Save settings only if they don't exist
        if (!get_option('maxbes_autolink_settings')) {
            add_option('maxbes_autolink_settings', $default_settings);
        }

        // Set plugin version
        add_option('maxbes_autolink_version', MAXBES_AUTOLINK_VERSION);

        // Clear any existing cache
        self::clear_all_cache();

        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Clear all plugin caches
     */
    private static function clear_all_cache() {
        global $wpdb;

        // Delete all transients starting with maxbes_autolink_
        $wpdb->query(
            "DELETE FROM {$wpdb->options}
            WHERE option_name LIKE '_transient_maxbes_autolink_%'
            OR option_name LIKE '_transient_timeout_maxbes_autolink_%'"
        );
    }
}
