<?php
/**
 * Helper functions for MaxBes Auto Link Manager
 *
 * @package MaxBes_Auto_Link_Manager
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Get plugin settings
 *
 * @return array
 */
function maxbes_autolink_get_settings() {
    $defaults = array(
        'enabled' => true,
        'max_links_per_post' => 12,
        'case_sensitive' => false,
        'protected_tags' => array('h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'a', 'code', 'pre', 'figcaption'),
        'cache_duration' => DAY_IN_SECONDS,
        'link_target' => '_self',
        'link_class' => 'maxbes-auto-link',
        'rel_attribute' => '',
        'selected_taxonomies' => array(),
        'selected_post_types' => array('post', 'page'),
    );

    $settings = get_option('maxbes_autolink_settings', array());
    return wp_parse_args($settings, $defaults);
}

/**
 * Update plugin settings
 *
 * @param array $settings New settings
 * @return bool
 */
function maxbes_autolink_update_settings($settings) {
    return update_option('maxbes_autolink_settings', $settings);
}

/**
 * Get all registered taxonomies (for selection in admin)
 *
 * @return array
 */
function maxbes_autolink_get_available_taxonomies() {
    $taxonomies = get_taxonomies(array('public' => true), 'objects');

    $available = array();
    foreach ($taxonomies as $taxonomy) {
        // Skip nav_menu and link_category
        if (in_array($taxonomy->name, array('nav_menu', 'link_category', 'post_format'))) {
            continue;
        }

        $available[$taxonomy->name] = array(
            'name' => $taxonomy->name,
            'label' => $taxonomy->label,
            'singular_name' => $taxonomy->labels->singular_name,
            'count' => wp_count_terms($taxonomy->name, array('hide_empty' => false))
        );
    }

    return $available;
}

/**
 * Get all public post types (for selection in admin)
 *
 * @return array
 */
function maxbes_autolink_get_available_post_types() {
    $post_types = get_post_types(array('public' => true), 'objects');

    $available = array();
    foreach ($post_types as $post_type) {
        // Skip attachments
        if ($post_type->name === 'attachment') {
            continue;
        }

        $count = wp_count_posts($post_type->name);

        $available[$post_type->name] = array(
            'name' => $post_type->name,
            'label' => $post_type->label,
            'singular_name' => $post_type->labels->singular_name,
            'count' => $count->publish
        );
    }

    return $available;
}

/**
 * Check if auto-linking is enabled
 *
 * @return bool
 */
function maxbes_autolink_is_enabled() {
    $settings = maxbes_autolink_get_settings();
    return !empty($settings['enabled']);
}

/**
 * Get priority for taxonomy or post type
 *
 * @param string $type Type (taxonomy name or post_type)
 * @return int Priority (1-10)
 */
function maxbes_autolink_get_default_priority($type) {
    // High priority taxonomies
    $high_priority = array('category', 'post_tag');

    // Medium priority
    $medium_priority = array('post', 'page');

    // Check if it's a property taxonomy (starts with property-)
    if (strpos($type, 'property-') === 0) {
        return 1; // Highest priority for property taxonomies
    }

    if (in_array($type, $high_priority)) {
        return 2;
    }

    if (in_array($type, $medium_priority)) {
        return 4;
    }

    return 5; // Default priority
}

/**
 * Format last import message
 *
 * @param int $count Number of imported items
 * @param string $type Type of import
 * @return string
 */
function maxbes_autolink_format_import_message($count, $type) {
    if ($count === 0) {
        return sprintf(__('No new %s to import.', 'maxbes-auto-link-manager'), $type);
    }

    return sprintf(
        _n(
            'Successfully imported %d keyword from %s.',
            'Successfully imported %d keywords from %s.',
            $count,
            'maxbes-auto-link-manager'
        ),
        $count,
        $type
    );
}

/**
 * Sanitize keyword data
 *
 * @param array $data Raw keyword data
 * @return array Sanitized data
 */
function maxbes_autolink_sanitize_keyword_data($data) {
    return array(
        'keyword' => isset($data['keyword']) ? sanitize_text_field($data['keyword']) : '',
        'url' => isset($data['url']) ? esc_url_raw($data['url']) : '',
        'link_type' => isset($data['link_type']) ? sanitize_text_field($data['link_type']) : 'custom',
        'source_type' => isset($data['source_type']) ? sanitize_text_field($data['source_type']) : null,
        'source_id' => isset($data['source_id']) ? absint($data['source_id']) : null,
        'priority' => isset($data['priority']) ? min(10, max(1, absint($data['priority']))) : 5,
        'max_links' => isset($data['max_links']) ? min(10, max(1, absint($data['max_links']))) : 1,
        'case_sensitive' => isset($data['case_sensitive']) ? (bool) $data['case_sensitive'] : false,
        'status' => isset($data['status']) ? sanitize_text_field($data['status']) : 'active',
    );
}

/**
 * Get uninstall data (for uninstall.php)
 *
 * @return array
 */
function maxbes_autolink_get_uninstall_data() {
    global $wpdb;

    return array(
        'table' => $wpdb->prefix . MAXBES_AUTOLINK_TABLE,
        'options' => array(
            'maxbes_autolink_settings',
            'maxbes_autolink_version',
        )
    );
}
