<?php
/**
 * Keywords tab template - CRUD interface
 *
 * @package MaxBes_Auto_Link_Manager
 */

if (!defined('ABSPATH')) {
    exit;
}

// Get current page for pagination
$paged = isset($_GET['paged']) ? max(1, absint($_GET['paged'])) : 1;
$per_page = 20;

// Get search query
$search = isset($_GET['s']) ? sanitize_text_field($_GET['s']) : '';

// Get filter
$filter_status = isset($_GET['filter_status']) ? sanitize_text_field($_GET['filter_status']) : 'all';

// Get keywords
if ($search) {
    $keywords = MaxBes_AutoLink_Database::search_keywords($search);
    $total = count($keywords);
} else {
    $keywords = MaxBes_AutoLink_Database::get_keywords_paginated($paged, $per_page, array('status' => $filter_status));
    $total = MaxBes_AutoLink_Database::get_keywords_count($filter_status);
}

$total_pages = ceil($total / $per_page);

// Check if editing
$editing = isset($_GET['edit']) ? absint($_GET['edit']) : 0;
$edit_keyword = $editing ? MaxBes_AutoLink_Database::get_keyword($editing) : null;
?>

<div class="maxbes-keywords-tab">

    <!-- Lite Version Upgrade Notice -->
    <div class="notice notice-info">
        <p>
            <strong><?php _e('Lite Version Limit:', 'maxbes-auto-link-manager'); ?></strong>
            <?php
            $current_count = MaxBes_AutoLink_Database::get_keywords_count('all');
            printf(
                __('You are using %d of %d keywords.', 'maxbes-auto-link-manager'),
                $current_count,
                MAXBES_AUTOLINK_MAX_KEYWORDS
            );
            ?>
            <a href="<?php echo esc_url(MAXBES_AUTOLINK_UPGRADE_URL); ?>" target="_blank" class="button button-primary button-small" style="margin-left: 10px;">
                <?php _e('Upgrade to Pro for Unlimited Keywords', 'maxbes-auto-link-manager'); ?>
            </a>
        </p>
    </div>

    <!-- Add/Edit Keyword Form -->
    <div class="maxbes-card">
        <h2>
            <span class="dashicons dashicons-plus-alt"></span>
            <?php echo $editing ? __('Edit Keyword', 'maxbes-auto-link-manager') : __('Add New Keyword', 'maxbes-auto-link-manager'); ?>
        </h2>

        <form method="post" class="maxbes-keyword-form">
            <?php wp_nonce_field('maxbes_autolink_action', 'maxbes_autolink_nonce'); ?>
            <input type="hidden" name="action" value="<?php echo $editing ? 'edit_keyword' : 'add_keyword'; ?>">
            <?php if ($editing): ?>
                <input type="hidden" name="keyword_id" value="<?php echo $editing; ?>">
            <?php endif; ?>

            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label><?php _e('Keyword', 'maxbes-auto-link-manager'); ?> <span class="required">*</span></label>
                    </th>
                    <td>
                        <input type="text"
                               name="keyword"
                               value="<?php echo $edit_keyword ? esc_attr($edit_keyword->keyword) : ''; ?>"
                               class="regular-text"
                               required
                               placeholder="<?php _e('e.g., Dubai, Luxury Villa, Investment', 'maxbes-auto-link-manager'); ?>">
                        <p class="description"><?php _e('The text to find and link in posts', 'maxbes-auto-link-manager'); ?></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row">
                        <label><?php _e('Target URL', 'maxbes-auto-link-manager'); ?> <span class="required">*</span></label>
                    </th>
                    <td>
                        <input type="url"
                               name="url"
                               value="<?php echo $edit_keyword ? esc_url($edit_keyword->url) : ''; ?>"
                               class="regular-text"
                               required
                               placeholder="<?php _e('https://example.com/page', 'maxbes-auto-link-manager'); ?>">
                        <p class="description"><?php _e('Where the keyword should link to', 'maxbes-auto-link-manager'); ?></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row">
                        <label><?php _e('Link Type', 'maxbes-auto-link-manager'); ?></label>
                    </th>
                    <td>
                        <select name="link_type">
                            <option value="custom" <?php echo ($edit_keyword && $edit_keyword->link_type === 'custom') ? 'selected' : ''; ?>><?php _e('Custom URL', 'maxbes-auto-link-manager'); ?></option>
                            <option value="taxonomy" <?php echo ($edit_keyword && $edit_keyword->link_type === 'taxonomy') ? 'selected' : ''; ?>><?php _e('Taxonomy Archive', 'maxbes-auto-link-manager'); ?></option>
                            <option value="post" <?php echo ($edit_keyword && $edit_keyword->link_type === 'post') ? 'selected' : ''; ?>><?php _e('Post', 'maxbes-auto-link-manager'); ?></option>
                            <option value="page" <?php echo ($edit_keyword && $edit_keyword->link_type === 'page') ? 'selected' : ''; ?>><?php _e('Page', 'maxbes-auto-link-manager'); ?></option>
                        </select>
                    </td>
                </tr>

                <!-- LITE VERSION: Priority field removed. Pro version offers advanced priority control -->

                <tr>
                    <th scope="row">
                        <label><?php _e('Max Links Per Post', 'maxbes-auto-link-manager'); ?></label>
                    </th>
                    <td>
                        <input type="number"
                               name="max_links"
                               value="<?php echo $edit_keyword ? esc_attr($edit_keyword->max_links) : '1'; ?>"
                               min="1"
                               max="10"
                               class="small-text">
                        <p class="description"><?php _e('Maximum occurrences to link in a single post', 'maxbes-auto-link-manager'); ?></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row">
                        <label><?php _e('Options', 'maxbes-auto-link-manager'); ?></label>
                    </th>
                    <td>
                        <label>
                            <input type="checkbox"
                                   name="case_sensitive"
                                   value="1"
                                   <?php echo ($edit_keyword && $edit_keyword->case_sensitive) ? 'checked' : ''; ?>>
                            <?php _e('Case Sensitive', 'maxbes-auto-link-manager'); ?>
                        </label>
                        <p class="description"><?php _e('Match exact capitalization only', 'maxbes-auto-link-manager'); ?></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row">
                        <label><?php _e('Status', 'maxbes-auto-link-manager'); ?></label>
                    </th>
                    <td>
                        <select name="status">
                            <option value="active" <?php echo ($edit_keyword && $edit_keyword->status === 'active') ? 'selected' : ''; ?>><?php _e('Active', 'maxbes-auto-link-manager'); ?></option>
                            <option value="inactive" <?php echo ($edit_keyword && $edit_keyword->status === 'inactive') ? 'selected' : ''; ?>><?php _e('Inactive', 'maxbes-auto-link-manager'); ?></option>
                        </select>
                    </td>
                </tr>
            </table>

            <p class="submit">
                <button type="submit" class="button button-primary">
                    <span class="dashicons dashicons-yes"></span>
                    <?php echo $editing ? __('Update Keyword', 'maxbes-auto-link-manager') : __('Add Keyword', 'maxbes-auto-link-manager'); ?>
                </button>
                <?php if ($editing): ?>
                    <a href="?page=maxbes-autolink&tab=keywords" class="button button-secondary">
                        <?php _e('Cancel', 'maxbes-auto-link-manager'); ?>
                    </a>
                <?php endif; ?>
            </p>
        </form>
    </div>

    <!-- Keywords List -->
    <div class="maxbes-card">
        <h2>
            <span class="dashicons dashicons-list-view"></span>
            <?php _e('Keywords List', 'maxbes-auto-link-manager'); ?>
            <span class="count">(<?php echo number_format($total); ?>)</span>
        </h2>

        <!-- LITE VERSION: Search and bulk actions removed. Available in Pro version -->
        <div class="notice notice-warning inline">
            <p>
                <?php _e('Want to search keywords and perform bulk operations?', 'maxbes-auto-link-manager'); ?>
                <a href="<?php echo esc_url(MAXBES_AUTOLINK_UPGRADE_URL); ?>" target="_blank"><?php _e('Upgrade to Pro', 'maxbes-auto-link-manager'); ?></a>
            </p>
        </div>

            <!-- Keywords Table -->
            <?php if (!empty($keywords)): ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php _e('Keyword', 'maxbes-auto-link-manager'); ?></th>
                            <th><?php _e('URL', 'maxbes-auto-link-manager'); ?></th>
                            <th><?php _e('Type', 'maxbes-auto-link-manager'); ?></th>
                            <th><?php _e('Max Links', 'maxbes-auto-link-manager'); ?></th>
                            <th><?php _e('Status', 'maxbes-auto-link-manager'); ?></th>
                            <th><?php _e('Actions', 'maxbes-auto-link-manager'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($keywords as $keyword): ?>
                            <tr>
                                <td>
                                    <strong><?php echo esc_html($keyword->keyword); ?></strong>
                                    <?php if ($keyword->source_type): ?>
                                        <br><small class="text-muted"><?php echo esc_html($keyword->source_type); ?></small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <a href="<?php echo esc_url($keyword->url); ?>" target="_blank" class="url-preview">
                                        <?php echo esc_html(substr($keyword->url, 0, 50)) . (strlen($keyword->url) > 50 ? '...' : ''); ?>
                                    </a>
                                </td>
                                <td><?php echo esc_html(ucfirst($keyword->link_type)); ?></td>
                                <td><?php echo $keyword->max_links; ?></td>
                                <td>
                                    <span class="status-badge status-<?php echo $keyword->status; ?>">
                                        <?php echo ucfirst($keyword->status); ?>
                                    </span>
                                </td>
                                <td class="actions-cell">
                                    <a href="?page=maxbes-autolink&tab=keywords&edit=<?php echo $keyword->id; ?>" class="button button-small">
                                        <?php _e('Edit', 'maxbes-auto-link-manager'); ?>
                                    </a>
                                    <a href="?page=maxbes-autolink&tab=keywords&action=delete&id=<?php echo $keyword->id; ?>&_wpnonce=<?php echo wp_create_nonce('delete_keyword_' . $keyword->id); ?>"
                                       class="button button-small button-link-delete"
                                       onclick="return confirm('<?php _e('Are you sure?', 'maxbes-auto-link-manager'); ?>');">
                                        <?php _e('Delete', 'maxbes-auto-link-manager'); ?>
                                    </a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>

                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                    <div class="tablenav bottom">
                        <div class="tablenav-pages">
                            <?php
                            echo paginate_links(array(
                                'base' => add_query_arg('paged', '%#%'),
                                'format' => '',
                                'prev_text' => __('&laquo; Previous'),
                                'next_text' => __('Next &raquo;'),
                                'total' => $total_pages,
                                'current' => $paged
                            ));
                            ?>
                        </div>
                    </div>
                <?php endif; ?>

            <?php else: ?>
                <p class="no-data">
                    <?php _e('No keywords found. Add your first keyword above or import from taxonomies.', 'maxbes-auto-link-manager'); ?>
                </p>
            <?php endif; ?>
    </div>
</div>
