/**
 * MaxBes Auto Link Manager - Admin JavaScript
 * @package MaxBes_Auto_Link_Manager
 */

(function($) {
    'use strict';

    $(document).ready(function() {

        /**
         * Select All Checkboxes
         */
        $('#cb-select-all').on('change', function() {
            const isChecked = $(this).is(':checked');
            $('input[name="keyword_ids[]"]').prop('checked', isChecked);
        });

        /**
         * Individual checkbox updates select all
         */
        $('input[name="keyword_ids[]"]').on('change', function() {
            const total = $('input[name="keyword_ids[]"]').length;
            const checked = $('input[name="keyword_ids[]"]:checked').length;
            $('#cb-select-all').prop('checked', total === checked);
        });

        /**
         * Bulk Actions Confirmation
         */
        $('#keywords-form').on('submit', function(e) {
            const bulkAction = $('select[name="bulk_action"]').val();
            const checkedItems = $('input[name="keyword_ids[]"]:checked').length;

            if (!bulkAction) {
                e.preventDefault();
                alert('Please select a bulk action.');
                return false;
            }

            if (checkedItems === 0) {
                e.preventDefault();
                alert('Please select at least one keyword.');
                return false;
            }

            if (bulkAction === 'delete') {
                if (!confirm(maxbesAutolink.confirm_bulk_delete)) {
                    e.preventDefault();
                    return false;
                }
            }
        });

        /**
         * Delete Keyword Confirmation
         */
        $('.button-link-delete').on('click', function(e) {
            if (!confirm(maxbesAutolink.confirm_delete)) {
                e.preventDefault();
                return false;
            }
        });

        /**
         * Form Validation
         */
        $('.maxbes-keyword-form').on('submit', function(e) {
            const keyword = $('input[name="keyword"]').val().trim();
            const url = $('input[name="url"]').val().trim();

            if (!keyword) {
                e.preventDefault();
                alert('Please enter a keyword.');
                $('input[name="keyword"]').focus();
                return false;
            }

            if (!url) {
                e.preventDefault();
                alert('Please enter a target URL.');
                $('input[name="url"]').focus();
                return false;
            }

            // Basic URL validation
            try {
                new URL(url);
            } catch (error) {
                e.preventDefault();
                alert('Please enter a valid URL (starting with http:// or https://).');
                $('input[name="url"]').focus();
                return false;
            }
        });

        /**
         * Import Form Validation
         */
        $('.maxbes-import-form').on('submit', function(e) {
            const checkedTaxonomies = $(this).find('input[type="checkbox"]:checked').length;

            if (checkedTaxonomies === 0) {
                e.preventDefault();
                alert('Please select at least one item to import.');
                return false;
            }

            // Show loading indicator
            const $button = $(this).find('button[type="submit"]');
            $button.prop('disabled', true);
            $button.html('<span class="dashicons dashicons-update-alt"></span> Importing...');

            // Allow form to submit
            return true;
        });

        /**
         * Settings Form - Select All Taxonomies
         */
        if ($('.taxonomy-grid').length) {
            // Add Select All / Deselect All buttons
            $('.taxonomy-grid').before(
                '<div class="taxonomy-select-actions" style="margin-bottom: 10px;">' +
                '<button type="button" class="button button-small taxonomy-select-all">Select All</button> ' +
                '<button type="button" class="button button-small taxonomy-deselect-all">Deselect All</button>' +
                '</div>'
            );

            $('.taxonomy-select-all').on('click', function() {
                $('.taxonomy-grid input[type="checkbox"]').prop('checked', true);
            });

            $('.taxonomy-deselect-all').on('click', function() {
                $('.taxonomy-grid input[type="checkbox"]').prop('checked', false);
            });
        }

        /**
         * Protected Tags - Select Recommended
         */
        if ($('.protected-tags-grid').length) {
            // Add recommended button
            $('.protected-tags-grid').before(
                '<div class="protected-tags-actions" style="margin-bottom: 10px;">' +
                '<button type="button" class="button button-small protected-select-recommended">Select Recommended</button>' +
                '</div>'
            );

            $('.protected-select-recommended').on('click', function() {
                const recommended = ['h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'a', 'code', 'pre', 'figcaption'];
                $('.protected-tags-grid input[type="checkbox"]').each(function() {
                    const value = $(this).val();
                    $(this).prop('checked', recommended.includes(value));
                });
            });
        }

        /**
         * Smooth Scroll to Form
         */
        if (window.location.hash === '#add-keyword') {
            $('html, body').animate({
                scrollTop: $('.maxbes-keyword-form').offset().top - 50
            }, 500);
        }

        /**
         * Clear Cache Confirmation
         */
        $('form').on('submit', function(e) {
            const action = $(this).find('input[name="action"]').val();

            if (action === 'clear_cache') {
                if (!confirm('Are you sure you want to clear all cache? This will temporarily slow down your site until cache is rebuilt.')) {
                    e.preventDefault();
                    return false;
                }
            }
        });

        /**
         * Copy URL to Clipboard
         */
        $('.url-preview').on('click', function(e) {
            e.preventDefault();
            const url = $(this).attr('href');

            // Try to copy to clipboard
            if (navigator.clipboard) {
                navigator.clipboard.writeText(url).then(function() {
                    // Show temporary success message
                    const $link = $(e.target);
                    const originalText = $link.text();
                    $link.text('URL Copied!');
                    setTimeout(function() {
                        $link.text(originalText);
                    }, 2000);
                });
            } else {
                // Fallback: open in new tab
                window.open(url, '_blank');
            }
        });

        /**
         * Taxonomy Checkbox Highlight
         */
        $('.taxonomy-checkbox, .import-checkbox').on('click', function() {
            const $checkbox = $(this).find('input[type="checkbox"]');
            $checkbox.prop('checked', !$checkbox.is(':checked'));
        });

        // Prevent checkbox double-toggle
        $('.taxonomy-checkbox input, .import-checkbox input').on('click', function(e) {
            e.stopPropagation();
        });

        /**
         * Auto-save Draft Keywords
         * Save form data to localStorage to prevent data loss
         */
        if ($('.maxbes-keyword-form').length) {
            const formId = 'maxbes_keyword_form_draft';

            // Load draft on page load
            const draft = localStorage.getItem(formId);
            if (draft && !$('input[name="keyword_id"]').val()) { // Only for new keywords
                const draftData = JSON.parse(draft);
                if (confirm('You have an unsaved draft. Would you like to restore it?')) {
                    Object.keys(draftData).forEach(function(key) {
                        $(`[name="${key}"]`).val(draftData[key]);
                    });
                }
            }

            // Save draft every 10 seconds
            let draftTimer;
            $('.maxbes-keyword-form input, .maxbes-keyword-form select').on('input change', function() {
                clearTimeout(draftTimer);
                draftTimer = setTimeout(function() {
                    const formData = {};
                    $('.maxbes-keyword-form').serializeArray().forEach(function(field) {
                        formData[field.name] = field.value;
                    });
                    localStorage.setItem(formId, JSON.stringify(formData));
                }, 10000);
            });

            // Clear draft on successful submit
            $('.maxbes-keyword-form').on('submit', function() {
                localStorage.removeItem(formId);
            });
        }

        /**
         * Statistics Charts (Future Enhancement)
         * Placeholder for chart.js integration
         */
        // if ($('.stat-box').length && typeof Chart !== 'undefined') {
        //     // Initialize charts here
        // }

        /**
         * Tooltips
         */
        $('.description, .help-tip').attr('title', function() {
            return $(this).text();
        });

        /**
         * Loading States
         */
        $('form[method="post"]').on('submit', function() {
            const $submitBtn = $(this).find('button[type="submit"], input[type="submit"]');
            if (!$submitBtn.is(':disabled')) {
                $submitBtn.prop('disabled', true);
                const originalText = $submitBtn.html();
                $submitBtn.data('original-text', originalText);
                $submitBtn.html('<span class="dashicons dashicons-update-alt" style="animation: rotation 1s infinite linear;"></span> Processing...');
            }
        });

        /**
         * Add CSS for rotating icon
         */
        if (!$('#maxbes-autolink-animations').length) {
            $('head').append(`
                <style id="maxbes-autolink-animations">
                    @keyframes rotation {
                        from { transform: rotate(0deg); }
                        to { transform: rotate(359deg); }
                    }
                </style>
            `);
        }

    });

})(jQuery);
