<?php
/**
 * Importer class
 * Handles importing keywords from taxonomies and posts
 *
 * @package MaxBes_Auto_Link_Manager
 */

if (!defined('ABSPATH')) {
    exit;
}

class MaxBes_AutoLink_Importer {

    /**
     * Import keywords from taxonomies
     *
     * @param array $taxonomies Array of taxonomy names
     * @return int Number of imported keywords
     */
    public function import_taxonomies($taxonomies) {
        $imported = 0;

        // Lite version: Only allow categories and tags
        $allowed_taxonomies = array('category', 'post_tag');

        foreach ($taxonomies as $taxonomy) {
            // Lite version: Skip if not category or tag
            if (!in_array($taxonomy, $allowed_taxonomies)) {
                continue;
            }

            $terms = get_terms(array(
                'taxonomy' => $taxonomy,
                'hide_empty' => false,
                'number' => 0, // Get all terms
            ));

            if (is_wp_error($terms) || empty($terms)) {
                continue;
            }

            foreach ($terms as $term) {
                // Lite version: Check keyword limit before importing
                $current_count = MaxBes_AutoLink_Database::get_keywords_count('all');
                if ($current_count >= MAXBES_AUTOLINK_MAX_KEYWORDS) {
                    return $imported; // Stop importing if limit reached
                }
                // Skip if keyword already exists
                if (MaxBes_AutoLink_Database::keyword_exists($term->name)) {
                    continue;
                }

                $data = array(
                    'keyword' => $term->name,
                    'url' => get_term_link($term),
                    'link_type' => 'taxonomy',
                    'source_type' => $taxonomy,
                    'source_id' => $term->term_id,
                    'priority' => maxbes_autolink_get_default_priority($taxonomy),
                    'max_links' => 1,
                    'case_sensitive' => 0,
                    'status' => 'active',
                );

                if (MaxBes_AutoLink_Database::insert_keyword($data)) {
                    $imported++;
                }
            }
        }

        return $imported;
    }

    /**
     * Import keywords from posts/pages
     * LITE VERSION: This feature is not available in the lite version
     *
     * @param array $post_types Array of post type names
     * @return int Number of imported keywords (always 0 in lite)
     */
    public function import_posts($post_types) {
        // Lite version: Post/page import not available
        return 0;
    }

    /**
     * Get count of importable items from taxonomies
     *
     * @param array $taxonomies Array of taxonomy names
     * @return int Count
     */
    public function count_importable_from_taxonomies($taxonomies) {
        $count = 0;

        // Lite version: Only count categories and tags
        $allowed_taxonomies = array('category', 'post_tag');

        foreach ($taxonomies as $taxonomy) {
            if (!in_array($taxonomy, $allowed_taxonomies)) {
                continue;
            }

            $terms = get_terms(array(
                'taxonomy' => $taxonomy,
                'hide_empty' => false,
                'fields' => 'count',
            ));

            if (!is_wp_error($terms)) {
                $count += $terms;
            }
        }

        return $count;
    }

    /**
     * Get count of importable items from post types
     * LITE VERSION: This feature is not available
     *
     * @param array $post_types Array of post type names
     * @return int Count (always 0 in lite)
     */
    public function count_importable_from_posts($post_types) {
        // Lite version: Post/page import not available
        return 0;
    }
}
