<?php
/**
 * Admin area controller
 * Handles admin menu, pages, and AJAX requests
 *
 * @package MaxBes_Auto_Link_Manager
 */

if (!defined('ABSPATH')) {
    exit;
}

class MaxBes_AutoLink_Admin {

    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            __('MaxBes Auto Link Manager', 'maxbes-auto-link-manager'),
            __('Auto Links', 'maxbes-auto-link-manager'),
            'manage_options',
            'maxbes-autolink',
            array($this, 'render_admin_page'),
            'dashicons-admin-links',
            80
        );
    }

    /**
     * Render main admin page
     */
    public function render_admin_page() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.'));
        }

        // Get current tab
        $current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'dashboard';

        // Handle form submissions
        $this->handle_form_submissions();

        // Include admin page template
        include MAXBES_AUTOLINK_PLUGIN_DIR . 'admin/views/admin-page.php';
    }

    /**
     * Handle form submissions
     */
    private function handle_form_submissions() {
        // Check nonce for security
        if (!isset($_POST['maxbes_autolink_nonce']) || !wp_verify_nonce($_POST['maxbes_autolink_nonce'], 'maxbes_autolink_action')) {
            return;
        }

        // Handle settings save
        if (isset($_POST['action']) && $_POST['action'] === 'save_settings') {
            $this->save_settings();
        }

        // Handle add keyword
        if (isset($_POST['action']) && $_POST['action'] === 'add_keyword') {
            $this->add_keyword();
        }

        // Handle edit keyword
        if (isset($_POST['action']) && $_POST['action'] === 'edit_keyword') {
            $this->edit_keyword();
        }

        // Handle delete keyword
        if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['id'])) {
            $this->delete_keyword();
        }

        // Handle bulk actions
        if (isset($_POST['bulk_action']) && isset($_POST['keyword_ids'])) {
            $this->handle_bulk_action();
        }

        // Handle import
        if (isset($_POST['action']) && $_POST['action'] === 'import_taxonomies') {
            $this->import_taxonomies();
        }

        if (isset($_POST['action']) && $_POST['action'] === 'import_posts') {
            $this->import_posts();
        }

        // Handle clear cache
        if (isset($_POST['action']) && $_POST['action'] === 'clear_cache') {
            $this->clear_cache();
        }
    }

    /**
     * Save settings
     */
    private function save_settings() {
        $settings = array(
            'enabled' => isset($_POST['enabled']) ? true : false,
            'max_links_per_post' => isset($_POST['max_links_per_post']) ? absint($_POST['max_links_per_post']) : 12,
            'case_sensitive' => isset($_POST['case_sensitive']) ? true : false,
            'protected_tags' => isset($_POST['protected_tags']) ? array_map('sanitize_text_field', $_POST['protected_tags']) : array(),
            'link_target' => isset($_POST['link_target']) ? sanitize_text_field($_POST['link_target']) : '_self',
            'link_class' => isset($_POST['link_class']) ? sanitize_text_field($_POST['link_class']) : 'maxbes-auto-link',
            'rel_attribute' => isset($_POST['rel_attribute']) ? sanitize_text_field($_POST['rel_attribute']) : '',
            'selected_taxonomies' => isset($_POST['selected_taxonomies']) ? array_map('sanitize_text_field', $_POST['selected_taxonomies']) : array(),
            'selected_post_types' => isset($_POST['selected_post_types']) ? array_map('sanitize_text_field', $_POST['selected_post_types']) : array('post', 'page'),
            'cache_duration' => DAY_IN_SECONDS,
        );

        if (maxbes_autolink_update_settings($settings)) {
            $this->add_notice(__('Settings saved successfully!', 'maxbes-auto-link-manager'), 'success');
        } else {
            $this->add_notice(__('Failed to save settings.', 'maxbes-auto-link-manager'), 'error');
        }
    }

    /**
     * Add keyword
     */
    private function add_keyword() {
        $data = maxbes_autolink_sanitize_keyword_data($_POST);

        // Lite version: Check keyword limit
        $current_count = MaxBes_AutoLink_Database::get_keywords_count('all');
        if ($current_count >= MAXBES_AUTOLINK_MAX_KEYWORDS) {
            $this->add_notice(
                sprintf(
                    __('Keyword limit reached! Lite version allows maximum %d keywords. <a href="%s" target="_blank">Upgrade to Pro</a> for unlimited keywords.', 'maxbes-auto-link-manager'),
                    MAXBES_AUTOLINK_MAX_KEYWORDS,
                    MAXBES_AUTOLINK_UPGRADE_URL
                ),
                'error'
            );
            return;
        }

        // Check if keyword already exists
        if (MaxBes_AutoLink_Database::keyword_exists($data['keyword'])) {
            $this->add_notice(__('Keyword already exists!', 'maxbes-auto-link-manager'), 'error');
            return;
        }

        $result = MaxBes_AutoLink_Database::insert_keyword($data);

        if ($result) {
            $this->add_notice(__('Keyword added successfully!', 'maxbes-auto-link-manager'), 'success');
        } else {
            $this->add_notice(__('Failed to add keyword.', 'maxbes-auto-link-manager'), 'error');
        }
    }

    /**
     * Edit keyword
     */
    private function edit_keyword() {
        $id = isset($_POST['keyword_id']) ? absint($_POST['keyword_id']) : 0;
        $data = maxbes_autolink_sanitize_keyword_data($_POST);

        // Check if keyword already exists (excluding current)
        if (MaxBes_AutoLink_Database::keyword_exists($data['keyword'], $id)) {
            $this->add_notice(__('Keyword already exists!', 'maxbes-auto-link-manager'), 'error');
            return;
        }

        $result = MaxBes_AutoLink_Database::update_keyword($id, $data);

        if ($result) {
            $this->add_notice(__('Keyword updated successfully!', 'maxbes-auto-link-manager'), 'success');
        } else {
            $this->add_notice(__('Failed to update keyword.', 'maxbes-auto-link-manager'), 'error');
        }
    }

    /**
     * Delete keyword
     */
    private function delete_keyword() {
        $id = isset($_GET['id']) ? absint($_GET['id']) : 0;
        $nonce = isset($_GET['_wpnonce']) ? $_GET['_wpnonce'] : '';

        if (!wp_verify_nonce($nonce, 'delete_keyword_' . $id)) {
            $this->add_notice(__('Security check failed.', 'maxbes-auto-link-manager'), 'error');
            return;
        }

        $result = MaxBes_AutoLink_Database::delete_keyword($id);

        if ($result) {
            $this->add_notice(__('Keyword deleted successfully!', 'maxbes-auto-link-manager'), 'success');
        } else {
            $this->add_notice(__('Failed to delete keyword.', 'maxbes-auto-link-manager'), 'error');
        }
    }

    /**
     * Handle bulk actions
     * LITE VERSION: Bulk operations not available
     */
    private function handle_bulk_action() {
        // Lite version: Bulk operations not available
        $this->add_notice(
            sprintf(
                __('Bulk operations are not available in Lite version. <a href="%s" target="_blank">Upgrade to Pro</a> to manage multiple keywords at once.', 'maxbes-auto-link-manager'),
                MAXBES_AUTOLINK_UPGRADE_URL
            ),
            'error'
        );
    }

    /**
     * Import taxonomies
     */
    private function import_taxonomies() {
        $taxonomies = isset($_POST['import_taxonomies']) ? array_map('sanitize_text_field', $_POST['import_taxonomies']) : array();

        if (empty($taxonomies)) {
            $this->add_notice(__('Please select at least one taxonomy.', 'maxbes-auto-link-manager'), 'error');
            return;
        }

        // Lite version: Only allow categories and tags
        $allowed = array('category', 'post_tag');
        $taxonomies = array_intersect($taxonomies, $allowed);

        if (empty($taxonomies)) {
            $this->add_notice(
                sprintf(
                    __('Lite version only supports Categories and Tags. <a href="%s" target="_blank">Upgrade to Pro</a> to import custom taxonomies.', 'maxbes-auto-link-manager'),
                    MAXBES_AUTOLINK_UPGRADE_URL
                ),
                'error'
            );
            return;
        }

        require_once MAXBES_AUTOLINK_PLUGIN_DIR . 'admin/class-autolink-importer.php';
        $importer = new MaxBes_AutoLink_Importer();
        $count = $importer->import_taxonomies($taxonomies);

        $current_count = MaxBes_AutoLink_Database::get_keywords_count('all');
        if ($current_count >= MAXBES_AUTOLINK_MAX_KEYWORDS) {
            $this->add_notice(
                sprintf(
                    __('Imported %d keywords. Keyword limit (%d) reached! <a href="%s" target="_blank">Upgrade to Pro</a> for unlimited keywords.', 'maxbes-auto-link-manager'),
                    $count,
                    MAXBES_AUTOLINK_MAX_KEYWORDS,
                    MAXBES_AUTOLINK_UPGRADE_URL
                ),
                'warning'
            );
        } else {
            $this->add_notice(sprintf(__('Imported %d keywords from taxonomies.', 'maxbes-auto-link-manager'), $count), 'success');
        }
    }

    /**
     * Import posts
     * LITE VERSION: Feature not available
     */
    private function import_posts() {
        // Lite version: Post/page import not available
        $this->add_notice(
            sprintf(
                __('Post/Page import is not available in Lite version. <a href="%s" target="_blank">Upgrade to Pro</a> to unlock this feature.', 'maxbes-auto-link-manager'),
                MAXBES_AUTOLINK_UPGRADE_URL
            ),
            'error'
        );
    }

    /**
     * Clear cache
     * LITE VERSION: Caching removed for better compatibility
     */
    private function clear_cache() {
        $this->add_notice(__('Lite version does not use caching.', 'maxbes-auto-link-manager'), 'info');
    }

    /**
     * Add admin notice
     */
    private function add_notice($message, $type = 'success') {
        set_transient('maxbes_autolink_admin_notice', array(
            'message' => $message,
            'type' => $type
        ), 30);
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        // Only load on our admin page
        if ($hook !== 'toplevel_page_maxbes-autolink') {
            return;
        }

        // CSS
        wp_enqueue_style(
            'maxbes-autolink-admin',
            MAXBES_AUTOLINK_PLUGIN_URL . 'admin/css/autolink-admin.css',
            array(),
            MAXBES_AUTOLINK_VERSION
        );

        // JavaScript
        wp_enqueue_script(
            'maxbes-autolink-admin',
            MAXBES_AUTOLINK_PLUGIN_URL . 'admin/js/autolink-admin.js',
            array('jquery'),
            MAXBES_AUTOLINK_VERSION,
            true
        );

        // Localize script
        wp_localize_script('maxbes-autolink-admin', 'maxbesAutolink', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('maxbes_autolink_ajax'),
            'confirm_delete' => __('Are you sure you want to delete this keyword?', 'maxbes-auto-link-manager'),
            'confirm_bulk_delete' => __('Are you sure you want to delete selected keywords?', 'maxbes-auto-link-manager'),
        ));
    }

    /**
     * AJAX handler stubs (will be implemented with JavaScript)
     */
    public function ajax_add_keyword() { /* Handled via form */ }
    public function ajax_edit_keyword() { /* Handled via form */ }
    public function ajax_delete_keyword() { /* Handled via form */ }
    public function ajax_bulk_action() { /* Handled via form */ }
    public function ajax_import_taxonomies() { /* Handled via form */ }
    public function ajax_import_posts() { /* Handled via form */ }
    public function ajax_clear_cache() { /* Handled via form */ }
}
